#include <iostream>
#include <cassert>
#include "heat_trad.h"
using namespace std;

namespace twong
{
	heat_trad::heat_trad(double Cutoff, double Dx, double Dt)
	{
		// Set the parameters.
		cutoff = Cutoff;
		dx = Dx;
		dt = Dt;

		// Calculate the multiplicative factor.
		set_factor();
	}

	heat_trad::~heat_trad()
	{
		// intentionally empty
	}

	void heat_trad::nextstep(pinned_growable_array& x_next, pinned_growable_array& x_current, int right_end_index)
	{
		// Check for stability of the method.
		assert(cutoff <= .5);

		// Make sure the arrays have the same starting index.
		assert(x_next.getStartingIndex() == x_current.getStartingIndex());

		// Make sure the right end index is greater than or equal to the starting index + 1.
		assert(right_end_index >= (x_next.getStartingIndex() + 1));

		// Run the traditional method.
		trad(x_next, x_current, right_end_index);
	}

	void heat_trad::set_cutoff(double Cutoff)
	{
		cutoff = Cutoff;
	}

	void heat_trad::set_dx(double Dx)
	{
		dx = Dx;

		// Calculate the new multiplicative factor.
		set_factor();
	}

	void heat_trad::set_dt(double Dt)
	{
		dt = Dt;

		// Calculate the new multiplicative factor.
		set_factor();
	}

	double heat_trad::get_dt()
	{
		return dt;
	}

	void heat_trad::trad(pinned_growable_array& x_next, pinned_growable_array& x_current, int right_end_index)
	{
		// Zero out the next mesh.
		x_next.zero();

		// The point behind the boundary stays the same.  This point is
		// the starting index.
		x_next[x_next.getStartingIndex()] = x_current[x_current.getStartingIndex()];

		// Starting at the point next to the starting index, solve the
		// heat equation to the right until we reach right_end_index.
		for (int i = x_next.getStartingIndex() + 1; i <= right_end_index; i++)
		{
			// Run the heat equation on the index.
			x_next[i] = factor * (x_current[i+1] - 2*x_current[i] + x_current[i-1]) + x_current[i];
		}

		// Now, run the heat equation after the right_end_index until
		// we drop below the cutoff.
		int i = right_end_index + 1;
		while (x_current[i] >= cutoff)
		{
			// Run the heat equation on the index.
			x_next[i] = factor * (x_current[i+1] - 2*x_current[i] + x_current[i-1]) + x_current[i];

			// Go to the next index
			++i;
		}

		// Run the heat equation on the point next to the cutoff.
		x_next[i] = factor * (x_current[i+1] - 2*x_current[i] + x_current[i-1]) + x_current[i];
	}

	void heat_trad::set_factor()
	{
		factor = dt / (dx*dx);
	}
} // namespace twong
